/* SCE CONFIDENTIAL
* PlayStation(R)3 Programmer Tool Runtime Library 430.001
* Copyright (C) 2012 Sony Computer Entertainment Inc.
* All Rights Reserved.
*/

#ifndef _SCE_COMMERCE_INTERFACE_H
#define _SCE_COMMERCE_INTERFACE_H

#include "np_toolkit_defines.h"


namespace sce { 
	namespace Toolkit {
		namespace NP {

	/// @brief
	/// The namespace containing PSN commerce functionality.
	///
	/// The namespace containing PSN commerce functionality.
	namespace Commerce{

	/// @brief
	/// External interface to the PSN commerce functionality.
	///
	/// This class contains the set of static methods for utilizing PSN commerce.
	class Interface
	{
	public:

		/// @brief 
		/// Creates a commerce session.
		///
		/// Creates a commerce session. When a commerce session has been created successfully, a
		/// commerceSessionCreated event will be sent to the %NP %Toolkit callback. This session is needed only
		/// before calling getCategoryInfo(), getProductList() or getDetailedProductInfo().
		///
		/// @retval	SCE_TOOLKIT_NP_SUCCESS						The operation was successful.
		/// @retval	SCE_TOOLKIT_NP_OFFLINE						The operation failed because the user is not connected to the network.  
		/// @retval	SCE_TOOLKIT_NP_NOT_INITIALISED				The operation failed because the %NP %Toolkit library is not initialized.
		static int createSession();

		/// @brief 
		/// Gets category information.
		///
		/// Gets category information that is available in the PlayStationStore.
		///
		/// @param 	info			A <c>Future</c> object, which receives the store category information retrieved from the PlayStationStore
		/// @param 	categoryId		The category that information is being requested for. Specify <c>NULL</c> to receive information for the root category.
		/// @param	async			A flag that indicates whether the function is non-blocking or blocking.
		///							Defaults to true so the function is non-blocking by default.
		///
		/// @retval					SCE_TOOLKIT_NP_SUCCESS				The operation was successful.
		/// @retval					SCE_TOOLKIT_NP_OFFLINE				The operation failed because the user is not connected to the network.  
		/// @retval					SCE_TOOLKIT_NP_INVALID_POINTER		The operation failed because the <c>info</c> pointer was invalid.
		/// @retval					SCE_TOOLKIT_NP_NOT_INITIALISED		The operation failed because the %NP %Toolkit library is not initialized.
		static int getCategoryInfo(sce::Toolkit::NP::Utilities::Future<CategoryInfo> *info, const char *categoryId, bool async = true);

		/// @brief 
		/// Gets a list of products that are available in the PlayStationStore.
		///
		/// Gets a list of products that are available in the PlayStationStore.
		///
		/// @param 	productList		A <c>Future</c> object, which receives the list of products that are available in the PlayStationStore.
		/// @param 	categoryId		The ID of category that the products are in. Specify <c>NULL</c> if the products are in the root category.
		///
		/// @param	async			Specifies whether the function will be called synchronously or asynchronously
		///
		/// @retval					SCE_TOOLKIT_NP_SUCCESS				The operation was successful.
		/// @retval					SCE_TOOLKIT_NP_OFFLINE				The operation failed because the user is not connected to the network.  
		/// @retval					SCE_TOOLKIT_NP_INVALID_POINTER		The operation failed because the <c>productList</c> pointer was invalid.
		/// @retval					SCE_TOOLKIT_NP_NOT_INITIALISED		The operation failed because the %NP %Toolkit library is not initialized.
		static int getProductList(sce::Toolkit::NP::Utilities::Future<std::vector<ProductInfo> > *productList, const char *categoryId, bool async = true);

		/// @brief 
		/// Gets detailed information about a product in the PlayStationStore.
		///
		/// Gets detailed information about a product in the PlayStationStore. This includes the product long description.
		///
		/// @param 	info			A <c>Future</c> object, which receives the detailed product information.
		/// @param 	productId		The ID of the product.
		/// @param 	categoryId		The ID of the category that the product is in. Specify <c>NULL</c> if the product is in the root category.
		///
		/// @param	async			A flag that indicates whether the function is non-blocking or blocking.
		///							Defaults to true so the function is non-blocking by default.
		///
		/// @retval					SCE_TOOLKIT_NP_SUCCESS				The operation was successful.
		/// @retval					SCE_TOOLKIT_NP_OFFLINE				The operation failed because the user is not connected to the network.  
		/// @retval					SCE_TOOLKIT_NP_INVALID_POINTER		The operation failed because the <c>info</c> pointer was invalid.
		/// @retval					SCE_TOOLKIT_NP_NOT_INITIALISED		The operation failed because the %NP %Toolkit library is not initialized.
		static int getDetailedProductInfo(sce::Toolkit::NP::Utilities::Future<ProductInfoDetailed> *info, const char *productId, const char *categoryId, bool async = true);
		
		/// @brief 
		/// Launches the PlayStationStore to a specified category where the user can browse and purchase.
		///
		/// Launches the PlayStationStore to a specified category where the user can browse and purchase.
		///	This causes the application to terminate. When the user has quit from the store, the application
		/// will re-start automatically.
		/// 
		/// @param 	categoryId		The Store category ID. Specify <c>NULL</c> to launch to the root category.
		/// @param  userData		The arbitrary data that can be obtained when rebooting game after store browsing.	
		///
		/// @retval					SCE_TOOLKIT_NP_SUCCESS				The operation was successful.
		/// @retval					SCE_TOOLKIT_NP_OFFLINE				The operation failed because the user is not connected to the network.  
		/// @retval					SCE_TOOLKIT_NP_NOT_INITIALISED		The operation failed because the %NP %Toolkit library is not initialized.
		///
		///	@note
		/// This is a synchronous function.
		static int categoryBrowse(const char *categoryId, const int &userData);

		/// @brief 
		/// Launches the PlayStationStore to a specified product where the user can purchase it.
		///
		/// Launches the PlayStationStore to a specified product where the user can purchase it.
		///	This can be either an in-game store overlay or the launch of the store
		/// after the termination of the application. In the latter case, the application
		/// will re-start automatically.
		///
		/// @param 	params			The product browse parameters
		/// @param	async			A flag that indicates whether the function is non-blocking or blocking.
		///							Defaults to true so the function is non-blocking by default.
		///
		/// @retval					SCE_TOOLKIT_NP_SUCCESS				The operation was successful.
		/// @retval					SCE_TOOLKIT_NP_OFFLINE				The operation failed because the user is not connected to the network.  
		/// @retval					SCE_TOOLKIT_NP_NOT_INITIALISED		The operation failed because the %NP %Toolkit library is not initialized.
		static int productBrowse(ProductBrowseParams &params, bool async = true);

		/// @brief 
		/// Upgrades a trial to full game. 
		///
		/// Upgrades a trial to full game. Launches to the PlayStationStore so the user can purchase the upgrade license package.
		///	The content ID of the license package must be specified in the PARAM.SFO.
		/// 
		/// @param	async			A flag that indicates whether the function is non-blocking or blocking.
		///							Defaults to true so the function is non-blocking by default.
		///
		/// @retval					SCE_TOOLKIT_NP_SUCCESS				The operation was successful.
		/// @retval					SCE_TOOLKIT_NP_OFFLINE				The operation failed because the user is not connected to the network.  
		/// @retval					SCE_TOOLKIT_NP_INVALID_POINTER		The operation failed because the <c>info</c> pointer was invalid.
		/// @retval					SCE_TOOLKIT_NP_NOT_INITIALISED		The operation failed because the %NP %Toolkit library is not initialized.
		static int upgradeTrial(bool async = true);

		/// @brief 
		/// Verifies if a trial has been upgraded to the full game.
		///
		/// Verifies if a trial has been upgraded to the full game. This is done by checking
		/// if the corresponding license file has been installed. This function can only be called synchronously.
		/// 
		/// @param 	upgraded		The result of the upgrade verification.
		/// @param 	contentId		The content ID of the license file to be checked
		///
		/// @retval					SCE_TOOLKIT_NP_SUCCESS				The operation was successful.
		/// @retval					SCE_TOOLKIT_NP_OFFLINE				The operation failed because the user is not connected to the network.  
		/// @retval					SCE_TOOLKIT_NP_INVALID_POINTER		The operation failed because the <c>info</c> pointer was invalid.
		/// @retval					SCE_TOOLKIT_NP_NOT_INITIALISED		The operation failed because the %NP %Toolkit library is not initialized.
		static int verifyTrialUpgrade(bool *upgraded, const char *contentId);

		/// @brief 
		/// Redeems a voucher code.
		///
		/// Redeems a voucher code. The function displays the voucher code input screen so a voucher/promotional code can be redeemed.
		/// 
		/// @param 	params			The voucher redemption parameters.
		/// @param	async			A flag that indicates whether the function is non-blocking or blocking.
		///							Defaults to true so the function is non-blocking by default.
		///
		/// @retval					SCE_TOOLKIT_NP_SUCCESS				The operation was successful.
		/// @retval					SCE_TOOLKIT_NP_OFFLINE				The operation failed because the user is not connected to the network.  
		/// @retval					SCE_TOOLKIT_NP_NOT_INITIALISED		The operation failed because the %NP %Toolkit library is not initialized.
		static int voucherCodeInput(VoucherInputParams &params, bool async = true);

		/// @brief 
		/// Gets a list of service entitlements.
		///
		/// Gets a list of service entitlements that are associated with the current PSN user.
		/// 
		/// @param 	list			A <c>Future</c> object, which receives the entitlement list.
		///
		/// @retval					SCE_TOOLKIT_NP_SUCCESS				The operation was successful.
		///
		/// @note
		/// This is an asynchronous function.
		static int getEntitlementList(sce::Toolkit::NP::Utilities::Future<std::vector<SceNpEntitlement> > *list);

		/// @brief
		/// Consumes an entitlement.
		///
		/// Consumes a specified amount of a consumable service entitlement.
		///
		/// @param 	id				The ID of the entitlement to consume.
		/// @param 	consumedCount	The amount to consume.
		/// @param	async			A flag that indicates whether the function is non-blocking or blocking.
		///							Defaults to true so the function is non-blocking by default.
		///
		/// @retval					SCE_TOOLKIT_NP_SUCCESS				The operation was successful.
		static int consumeEntitlement(const SceNpEntitlementId &id, const uint32_t &consumedCount, bool async = true);

		/// @brief
		/// Displays the checkout dialog.
		///
		/// Displays the checkout dialog to the user where they can purchase a list of SKUs.
		///
		/// @param	params			The checkout input params containing a list of SKUs to be displayed and a memory container.
		/// @param	async			A flag that indicates whether the function is non-blocking or blocking.
		///							Defaults to true so the function is non-blocking by default.
		///
		/// @retval					SCE_TOOLKIT_NP_SUCCESS				The operation was successful.
		static int checkout(CheckoutInputParams &params, bool async = true);

		/// @brief
		/// Displays the download list dialog.
		///
		/// Displays the download list dialog to the user where they can re-download from a list of purchased SKUs.
		///
		/// @param	params			The download list input params containing a list of SKUs to be displayed and a memory container. Specify an empty list to see full list of items available.
		/// @param	async			A flag that indicates whether the function is non-blocking or blocking.
		///							Defaults to true so the function is non-blocking by default.
		///
		/// @retval					SCE_TOOLKIT_NP_SUCCESS				The operation was successful.
		static int displayDownloadList(DownloadListInputParams &params, bool async = true);
	};

			}//end Commerce
		}// end NP
	}//end Toolkit
}//end Sce
#endif
